;*****************************************************************************        
;
;   Module:     multiplex.asm
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 6/6/08                                            
;
;               Demonstration of SPI functions to communicate with one or more
;               MCP23S17 port expander IC's. The demo shows four devices being
;               used to drive a 32 x 32 LED matrix. The circuit diagram and 
;               full explaination can be found in the PicNMix column.
;
;*****************************************************************************        


	LIST P=18F2520
	
	#INCLUDE <P18CXXX.INC>
	#include "config.inc"


; Use Equates directives to give constants meaningful names


; Do not change these two definitions
SPI_USE_HW_MODULE       EQU 0
SPI_USE_BIT_BASHED      EQU 1


; ***** NOTE ******
; This definition is used to determine if the SPI interface 
; uses the inbuilt module or is bit bashed. Select one of
; the SPI_USE defintions above.

SPI_MODULE_MODE         EQU SPI_USE_BIT_BASHED




;*****************************************************************************        
;
; SPI Bus hardware constants
;
;*****************************************************************************        
SSPSTAT_VAL             EQU 0x00    ; 00000000
SSPCON1_VAL             EQU 0x30    ; 00110000

PORTC_APP_TRIS_VAL      EQU 0xE3    ; 11100011
DEVICE_CS_BIT           EQU 2       ; The bit number, used in port bit operations
DEVICE_SCK_BIT          EQU 3       ; The bit number, used in port bit operations
DEVICE_SI_BIT           EQU 5       ; The bit number, used in port bit operations
DEVICE_SO_BIT           EQU 4       ; The bit number, used in port bit operations



;*****************************************************************************        
;
; RAM Variables
;
;*****************************************************************************        

devAddr                 EQU 0x11    ; The device address ( 0 - 7 )
devRegAddr              EQU 0x12    ; The register to access ( 0 - FF )
devData                 EQU 0x13    ; buffer for data written to / read from device
bbTX                    EQU 0x14    ; Transmit shift register for bit-bashing output
bbRX                    EQU 0x15    ; Receive shift register for bit-bashing input
bbCount                 EQU 0x16    ; Counts the 8 clocks for bit bashing
delay1			    EQU	0x17    ; used to time row delay
delay2			    EQU	0x18	; used to time row delay
imageRow			    EQU	0x19    ; Holds a rotating bit field representing the rows
									; This is four bytes, with a zero bit indicating
									; the active row


;*****************************************************************************        
;
; Image Buffer
; 
;  The data to be displayed is stored in 128 bytes of RAM, held in BANK1
;  Normal variable ( in the preceeding section ) are located in BANK0, and
;  accessed using the 'Access Bank' feature which is so handy on PIC18F parts.
;  4 consequtive bytes hold the first row of the display. The MSB of the first 
;  byte is the top left most pixel; the LSB of the forth byte is the right most 
;  pixel. 
;
;*****************************************************************************        
imageBuff               EQU 0x00    


;*****************************************************************************        
;
;  Code follows
;
;*****************************************************************************        


	ORG 0
	goto	MAIN


;*****************************************************************************        
;
;   Macro :  HBIT_DELAY
;            Helper macro that pauses for ONE HALF of an SPI bit period
;            This is used by the bit-bashing implementation of SPI
;            As SPI is a synchronous protocol you can decide on your own clock
;            speed, up to the limit of the device. This version runs at about 
;            500KHz on a 4MHz CPU clock.
;
;   Input:   None
;
;   Output:  None
;
;*****************************************************************************        
HBIT_DELAY  macro  
            nop
            endm
            
            

;*****************************************************************************        
;
;   Function:  TXRX_SPI
;               Performs the low level SPI functionality;
;               Transmitting a byte and simultaneously receiving a byte
;   
;               SPI is a funny protocol. As it has dedicated transmit and 
;               receive lines, it can transmit a byte and receive one at the
;               same time. But few devices use this facility.
;               So when you are sending a byte, rubbish is received.
;               And when you want to receive a byte, you actually have to transmit
;               a byte, even though it is unused. Thats just the way it works!
;
;               This call blocks while it waits for the transfer to complete
;               As it uses the built in SPI module, you could set the CPU
;               up to interrupt when done, enabling you to do other stuff
;               rather than just wait. Blocking simplifies the example.
;
;               This function will implement either Bit-Bashed or SPI Module
;               communications, depending on the setting of the SPI_MODULE_MODE
;               symbol at the top of the file
;               
;
;   Input:      Value to transmit in W
;   
;   Output:     Value received in W
;
;*****************************************************************************        
TXRX_SPI 
    IF (SPI_MODULE_MODE == SPI_USE_HW_MODULE) 
    
    ; Use the HW module to send and receive the data 
    movwf   SSPBUF              ; send data
    btfss   SSPSTAT, BF         ; wait for transfer to complete
    bra     $-2
    movf    SSPBUF, W           ; Get data
    
    ELSE
    
    ; Use bit bashing to get the byte out and data in
    movwf   bbTX
    clrf    bbRX
    movlw   8
    movwf   bbCount
    
SPILoop    
    ; Transmit a databit, most significant first
    rlcf    bbTX,F
    btfss   STATUS, C
    bcf     LATC, DEVICE_SO_BIT
    btfsc   STATUS, C
    bsf     LATC, DEVICE_SO_BIT
    
    bsf     LATC, DEVICE_SCK_BIT ; Assert the clock ( drive to 1 )
    
    HBIT_DELAY                  ; delay for a 1/2 clock period
    
    ; Now sample the input signal, and store in bbRX
    rlcf    bbRX,F
    btfss   PORTC, DEVICE_SI_BIT
    bcf     bbRX, 0
    btfsc   PORTC, DEVICE_SI_BIT
    bsf     bbRX, 0

    bcf     LATC, DEVICE_SCK_BIT ; De-assert the clock ( drive to 0 )
        
    HBIT_DELAY                  ; delay for a 1/2 clock period
        
    decfsz  bbCount,F
    bra     SPILoop
    movf    bbRX, W
    
    ENDIF
    
    return



;*****************************************************************************        
;
;   Function:  TX_SPI
;               Identical to TXRX_SPI, except it only transmits data
;               
;
;   Input:      Value to transmit in W
;   
;   Output:     None
;
;*****************************************************************************        
TX_SPI 
    IF (SPI_MODULE_MODE == SPI_USE_HW_MODULE) 
    
    ; Use the HW module to send and receive the data 
    movwf   SSPBUF              ; send data
    btfss   SSPSTAT, BF         ; wait for transfer to complete
    bra     $-2
    movf    SSPBUF, W           ; Get data
    
    ELSE
    
    ; Use bit bashing to get the byte out
    movwf   bbTX
    movlw   8
    movwf   bbCount
    
SPITLoop    
    ; Transmit a databit, most significant first
    rlcf    bbTX,F
    btfss   STATUS, C
    bcf     LATC, DEVICE_SO_BIT
    btfsc   STATUS, C
    bsf     LATC, DEVICE_SO_BIT
    
    bsf     LATC, DEVICE_SCK_BIT ; Assert the clock ( drive to 1 )
    
    HBIT_DELAY                  ; delay for a 1/2 clock period
    
    bcf     LATC, DEVICE_SCK_BIT ; De-assert the clock ( drive to 0 )
        
    HBIT_DELAY                  ; delay for a 1/2 clock period
        
    
    decfsz  bbCount,F
    bra     SPITLoop
    
    ENDIF
    
    return





;*****************************************************************************        
;
;   Function :  SPIHWInit
;               configures the SPI peripherial hardware into mode 1,1
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
SPIHWInit

    ; Configure the Chip Select pin as an output, and set it to disabled (high)
    movlw   PORTC_APP_TRIS_VAL
    movwf   TRISC 
    bsf     LATC, DEVICE_CS_BIT

    IF (SPI_MODULE_MODE == SPI_USE_HW_MODULE) 

    ; Setup the HW module to send and receive the data
    movlw   SSPSTAT_VAL
    movwf   SSPSTAT
    
    movlw   SSPCON1_VAL
    movwf   SSPCON1
   
    ELSE
    
    ; We are using bit-bashing, so setup the IO pins and disable the SPI Module

    bcf     SSPCON1, SSPEN      ; Disable SPI module

    bcf     LATC, DEVICE_SCK_BIT
        
    ENDIF
    
    return


;*****************************************************************************        
;
;                          MCP23S17 Functions follow
;
;*****************************************************************************        

	
;*****************************************************************************        
;
;   Function :  MCP23S17Init
;               Configures the four devices for the multiplex display.
;               This just means:
;                Enable hardware address detection,
;                Configuring all 64 I/O pins as outputs,
;                Set them all low.
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        

MCP23S17Init
	; Enable hardware address detection. 
	movlw	0
	movwf	devAddr
	movlw	0x0A
	movwf	devRegAddr
	movlw	0x08
	movwf	devData
	call 	MCP23S17WriteByte
	
	; Enable all I/O pins as outputs, and set them high
	; to turn off the display
	; Loop round the four IC's, starting at device # 0
	movlw	0
	movwf	devAddr
	
minit01
	movlw	0				
	movwf	devRegAddr
	movlw	0				
	movwf	devData
	call 	MCP23S17WriteByte

	movlw	1
	movwf	devRegAddr
	movlw	0
	movwf	devData
	call 	MCP23S17WriteByte

	movlw	0x14
	movwf	devRegAddr
	movlw	0xFF
	movwf	devData
	call 	MCP23S17WriteByte
	
	movlw	0x15
	movwf	devRegAddr
	movlw	0xFF
	movwf	devData
	call 	MCP23S17WriteByte
		
	; Stop when we have done all four devices.
	; Otherwise, loop back and do the next device
	incf	devAddr, F
	movf	devAddr, W
	sublw	0x04
	btfss	STATUS, Z
	goto	minit01

	return


;*****************************************************************************        
;
;   Function :  HWInit
;               Configures the various functions of the processor to allow us 
;               to use it in this particular application
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        

HWInit
	movlw	0x70	
	movwf	OSCCON			; Set the internal oscillator to 8MHz
	movlw	0x40
	movwf	OSCTUNE			; Enable the PLL, giving a CPU clock of 32MHz
	call	StartupDelay	; Give the port expander ICs time to power up
	call	SPIHWInit		; Configure the IO pins for the SPI bus
	return
	


;*****************************************************************************        
;
;   Function :  MCP23S17ReadByte
;               This is a global function for reading data from the device
;
;   Input:      Device Number in devAddr ( 0 - 7 ) 
;   Input:      Register Number in devRegAddr ( 0 - FF. not all implemented )
;               
;
;   Output:     Received byte in devData
;
;*****************************************************************************        
MCP23S17ReadByte
    bcf     LATC, DEVICE_CS_BIT ; Enable device
    movf	devAddr, W
    addwf	devAddr, W			; Shift the device address one to the left
    iorlw   0x41				; Set fixed bits for a read   

    call    TXRX_SPI            ; Tx address byte, dummy read
    
    movf    devRegAddr, W
    call    TXRX_SPI            ; Tx register byte, dummy read

    call    TXRX_SPI            ; Tx dummy byte, read data
    movwf   devData
   
    bsf     LATC, DEVICE_CS_BIT ; Disable device

    return



;*****************************************************************************        
;
;   Function :  MCP23S17WriteByte
;               This is a global function for writing data to the device
;
;   Input:      Device Number in devAddr ( 0 - 7 ) 
;   Input:      Register Number in devRegAddr ( 0 - FF. not all implemented )
;   Input:      Data to write in devData ( 0 - FF )
;               
;
;   Output:     None
;
;*****************************************************************************        
MCP23S17WriteByte
    bcf     LATC, DEVICE_CS_BIT ; Enable device
    movf	devAddr, W
    addwf	devAddr, W			; Shift the device address one to the left
    iorlw   0x40				; Set fixed bits for a read   

    call    TXRX_SPI            ; Tx address byte, dummy read
    
    movf    devRegAddr, W
    call    TXRX_SPI            ; Tx register byte, dummy read

    movf    devData, W
    call    TXRX_SPI            ; Tx data byte, dummy read

   
    bsf     LATC, DEVICE_CS_BIT ; Disable device

    return


;*****************************************************************************        
;
;   Function :  DisplayRow
;               Transfers the contents of the row of data from the image
;               buffer onto the LEDs
;
;   Input:      required row in imageRow
;               
;
;   Output:     None. LEDs updated
;
;*****************************************************************************        

DisplayRow
	; We do the writing to the devices as quickly as possible here, to 
	; reduce display flicker.
	
	; Turn off the row data, so we display nothing as we move to the new row
    bcf     LATC, DEVICE_CS_BIT ; Enable device
	movlw	0x40
	call    TX_SPI
	movlw	0x14
	call    TX_SPI
	movlw	0xFF
	call    TX_SPI
	movlw	0xFF
	call    TX_SPI
    bsf     LATC, DEVICE_CS_BIT ; Disable device
    nop
    bcf     LATC, DEVICE_CS_BIT ; Enable device
	movlw	0x42
	call    TX_SPI
	movlw	0x14
	call    TX_SPI
	movlw	0xFF
	call    TX_SPI
	movlw	0xFF
	call    TX_SPI
    bsf     LATC, DEVICE_CS_BIT ; Disable device
	nop
		
    bcf     LATC, DEVICE_CS_BIT ; Enable device
	movlw	0x44
	call    TX_SPI
	movlw	0x14
	call    TX_SPI
	movf	imageRow, W
	call    TX_SPI
	movf	imageRow+1, W
	call    TX_SPI
    bsf     LATC, DEVICE_CS_BIT ; Disable device
    nop
    bcf     LATC, DEVICE_CS_BIT ; Enable device
	movlw	0x46
	call    TX_SPI
	movlw	0x14
	call    TX_SPI
	movf	imageRow+2, W
	call    TX_SPI
	movf	imageRow+3, W
	call    TX_SPI
    bsf     LATC, DEVICE_CS_BIT ; Disable device
	nop
	
	; Write the new row's data to the display
    bcf     LATC, DEVICE_CS_BIT ; Enable device
	movlw	0x40
	call    TX_SPI
	movlw	0x14
	call    TX_SPI
	movf	POSTINC0, W
	call    TX_SPI
	movf	POSTINC0, W
	call    TX_SPI
    bsf     LATC, DEVICE_CS_BIT ; Disable device
    nop
    bcf     LATC, DEVICE_CS_BIT ; Enable device
	movlw	0x42
	call    TX_SPI
	movlw	0x14
	call    TX_SPI
	movf	POSTINC0, W
	call    TX_SPI
	movf	POSTINC0, W
	call    TX_SPI
    bsf     LATC, DEVICE_CS_BIT ; Disable device
	
	return

	
	
;*****************************************************************************        
;
;   Function :  ShowRowPause
;               A simple delay, to allow the current state of the LEDs to be
;               shown long enough to register in the eyes.
;               This delay should be 1ms, and the loop times are based on 8MHz
;               clock speed.
;
;   Input:      None.
;               
;
;   Output:     None.
;
;*****************************************************************************        

ShowRowPause

	movlw	0x09
	movwf	delay2
srp000
	movlw	0x80
	movwf	delay1
srp001
	decfsz	delay1
	goto	srp001
	decfsz	delay2
	goto	srp000
	return
	


;*****************************************************************************        
;
;   Function :  StartupDelay
;               A simple delay called at startup, to allow the reset to the 
;               port expander IC's to occur. This is about 30ms
;
;   Input:      None.
;               
;
;   Output:     None.
;
;*****************************************************************************        

StartupDelay
	movlw	0x00
	movwf	delay2
sd000
	movlw	0x00
	movwf	delay1
sd001
	decfsz	delay1
	goto	sd001
	decfsz	delay2
	goto	sd000
	return
	
	

;*****************************************************************************        
;
;   Function :  InitPanelData
;               This routine loads the display buffer with the image
;               to be displayed. That data can come from anywhere - 
;               flash memory, another device, or perhaps calculated.
;               For this example we will load a simple pattern into the
;               top three lines
;
;   Input:      None
;               
;
;   Output:     None.
;
;*****************************************************************************        

InitPanelData
	; Select BANK1 where the image buffer lives
	movlw	0x01
	movwf	BSR
	
	movlw	0x55
	movwf	imageBuff, 1
	movwf	imageBuff+1, 1
	movwf	imageBuff+2, 1
	movwf	imageBuff+3, 1
	movlw	0xAA
	movwf	imageBuff+4, 1
	movwf	imageBuff+5, 1
	movwf	imageBuff+6, 1
	movwf	imageBuff+7, 1
	movlw	0x55
	movwf	imageBuff+8, 1
	movwf	imageBuff+9, 1
	movwf	imageBuff+0x0A, 1
	movwf	imageBuff+0x0B, 1
	
	clrf	BSR
	return



;*****************************************************************************        
;
;   Function :  Main
;               This is the main entry point
;
;   Input:      None
;               
;
;   Output:     None. Never exits.
;
;*****************************************************************************        

MAIN
	call	HWInit			; Setup the general CPU features
	call	MCP23S17Init	; Configure the device driver
	
	call	InitPanelData	; Load the display buffer
	
	; Configure the 4 byte row mask. A 0 means active row.
	movlw	0xFE
	movwf	imageRow
	movlw	0xFF
	movwf	imageRow+1
	movwf	imageRow+2
	movwf	imageRow+3

	movlw	0x01
	movwf	FSR0H			; Sets up a pointer to the display buffer
m000
	clrf	FSR0L			; Use FSR0 to point into the display buffer
	
m001
	call	DisplayRow
	call	ShowRowPause	; Delay, to allow row to be visible
	
	; Shift the row mask to the next row
	
	rlcf	imageRow, F
	rlcf	imageRow+1, F
	rlcf	imageRow+2, F
	rlcf	imageRow+3, F
	btfsc	STATUS,C
	bsf		imageRow, 0	
	
	movf	FSR0L, W
	sublw	0x80
	btfss	STATUS, Z
	goto	m001			; display next row
	goto	m000			; Start again from the beginning
	
    
	END